import ast

def parse_arguments(node):
    if isinstance(node, ast.Name):
        return node.id
    elif isinstance(node, ast.List):
        return [parse_arguments(item) for item in node.elts]
    elif isinstance(node, ast.Constant):
        return node.value
    elif isinstance(node, ast.Call):
        return parse_function_call(node)
    else:
        raise ValueError(f"Unsupported node type: {node}")

def parse_function_call(node):
    function_name = node.func.id
    args = [parse_arguments(arg) for arg in node.args]
    kwargs = {keyword.arg: parse_arguments(keyword.value) for keyword in node.keywords}
    return {"function_name": function_name, "arguments": args, "keyword_arguments": kwargs}

def parse_function_call_string(function_call):
    try:
        parsed_function = ast.parse(function_call.strip(), mode='eval')
        return parse_function_call(parsed_function.body)
    except Exception as e:
        print(f"Error parsing function call: {e}")
        return None

def build_function_call(parsed_function):
    function_name = parsed_function["function_name"]
    args = parsed_function["arguments"]
    kwargs = parsed_function["keyword_arguments"]

    # Build the arguments string
    args_string = ", ".join(str(arg) for arg in args)

    # Build the keyword arguments string
    kwargs_string = ", ".join(f"{key}={value}" for key, value in kwargs.items())

    # Combine args and kwargs into the final function call string
    if args:
        function_call_string = f"{function_name}({args_string}, {kwargs_string})"
    else:
        function_call_string = f"{function_name}({kwargs_string})"
    
    return function_call_string

def test():
    # Example usage
    function_call = "conditional_format_columns(columns = columns, conditionalFormatType = \"Preset\", presetCriterion = \"belowAverage\", fontColor=\"#F8696B\", bold=True)"
    parsed_function = parse_function_call_string(function_call)

    if parsed_function:
        print(parsed_function)

if __name__ == "__main__":
    test()